--# ***********************************************************************
--# **  PROJECT: eLAB, 021d, implement RealTimeClock
--# **  PURPOSE: Design implementation
--# **  FILENAME: RTC_CPLD1.vhd
--# **  FILETYPE: Main VHDL file
--# **  COMPILER: XILINX ISE 6.2.03i
--# ***********************************************************************
--# **  Created by: Peter Seng, SENG digitale Systeme GmbH
--# **	 Im Bruckwasen 35, D-73037 Goeppingen, Germany, http://www.seng.de
--# **	 Copyright (C) 2003 SENG digitale Syteme GmbH
--# **  Last changes: 2004-06-18
--# ***********************************************************************


---------------------------------------------------
-- Behavioural of Real Time Clock, implementation 1
---------------------------------------------------
--
--	Logic realized in CPLD, 4-digit 7-segment common cathode LED display, 4 input buttons, speaker:
--
--	Used hardware:
--		eLAB eMOD-d (CPLD, xc9572xl)
--		eLAB eMOD-b (4-digit 7-segment common cathode display, speaker)
--		eLAB eMOD-c (input buttons and LEDs)
--
--	Display:
--		DS_data connected to X1, P1..P8
--		DS_enable connected to X2, P9..P12
--		DS1..DS4 --> Time
--		DS3, DP --> always enabled
--
-- Speaker:
--		connected to X2, P16
--		enabled if SW4 pressed
--
--	Buttons:
--		connected to X3, P17..P20
--		SW1 --> (AUTO)INCREMENT
--		SW2 --> (AUTO)DECREMENT
--		SW3 --> ENABLE TIME SETTING
--		SW4 --> ENABLE SPEAKER 
--		(SW3 and SW1) or (SW3 and SW2) --> SET TIME
--
---------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;


--  Uncomment the following lines to use the declarations that are
--  provided for instantiating Xilinx primitive components.
library UNISIM;
use UNISIM.VComponents.all;


entity RTC_CPLD1 is
	Port ( 
		-- DEBUG ONLY
		--
		-- END OF DEBUG ONLY

		clk: in STD_LOGIC; -- 4MHz0000 crystal
		nReset: in STD_LOGIC; -- nReset input
		P1,P2,P3,P4,P5,P6,P7,P8: out STD_LOGIC; -- display, DS_data
		P9,P10,P11,P12: out STD_LOGIC; -- display, DS_enable
		P16: out STD_LOGIC; -- speaker
		P17,P18,P19,P20: in STD_LOGIC -- button, SW1..SW4
	);
end RTC_CPLD1;


architecture Behavioral of RTC_CPLD1 is
	attribute loc: string;

	-- I/O pin assignments for eLAB, eMOD-d (CPLD, xc9572xl):
	attribute loc of clk: signal is "P5";
	attribute loc of nReset: signal is "P39";
	attribute loc of P1: signal is "P6";	-- connector X1, P1...P8
	attribute loc of P2: signal is "P42";
	attribute loc of P3: signal is "P27";
	attribute loc of P4: signal is "P26";
	attribute loc of P5: signal is "P24";
	attribute loc of P6: signal is "P28";
	attribute loc of P7: signal is "P25";
	attribute loc of P8: signal is "P22";
	attribute loc of P9: signal is "P19"; -- connector X2, P9...P16
	attribute loc of P10: signal is "P18";
	attribute loc of P11: signal is "P14";
	attribute loc of P12: signal is "P20";
	--	attribute loc of P13: signal is "P11";
	--	attribute loc of P14: signal is "P13";
	--	attribute loc of P15: signal is "P12";
	attribute loc of P16: signal is "P9";
	attribute loc of P17: signal is "P8"; -- connector X3, P17...P24
	attribute loc of P18: signal is "P4";
	attribute loc of P19: signal is "P3";
	attribute loc of P20: signal is "P2";
	--	attribute loc of P21: signal is "P1";
	--	attribute loc of P22: signal is "P44";
	--	attribute loc of P23: signal is "P43";
	--	attribute loc of P24: signal is "P38";
	--	attribute loc of P25: signal is "P7"; -- connector X4, P25...P32
	--	attribute loc of P26: signal is "P40";
	--	attribute loc of P27: signal is "P37";
	--	attribute loc of P28: signal is "P36";
	--	attribute loc of P29: signal is "P35"; -- X4 dual use !!!, JTAG TCK - J5
	--	attribute loc of P30: signal is "P33"; -- X4 dual use !!!, JTAG TMS - J6
	--	attribute loc of P31: signal is "P34"; -- X4 dual use !!!, JTAG TDO - J7
	--	attribute loc of P32: signal is "P29"; -- X4 dual use !!!, JTAG TDI - J8

 	-- signal definitions:
	signal COUNT: STD_LOGIC_VECTOR(27 downto 0); -- base counter 1/60 Hz
	signal MinL: STD_LOGIC_VECTOR(3 downto 0); -- time LSB digit 1, minutes, 0..9
	signal MinH: STD_LOGIC_VECTOR(2 downto 0); -- time digit 2, minutes x 10, 0..5
	signal HourL: STD_LOGIC_VECTOR(3 downto 0); -- time digit 3, hours, 0..9
	signal HourH: STD_LOGIC_VECTOR(1 downto 0); -- time MSB digit 4, hours x 10, 0..2
	signal MinHinc, MinHdec, HourLinc, HourLdec, HourHinc, HourHdec: STD_LOGIC; -- time carry
	signal DS_dataBin: STD_LOGIC_VECTOR(3 downto 0); -- display data, binary
	signal DS_data: STD_LOGIC_VECTOR(6 downto 0); -- display data, 7-segment data
	signal DS_DP: STD_LOGIC; -- display, decimal point
	signal DS_enable: STD_LOGIC_VECTOR(3 downto 0); -- display data
	signal SW1, SW2, SW3, SW4: STD_LOGIC; -- input buttons
	signal DebounceSR: STD_LOGIC_VECTOR(1 downto 0); -- input buttons debounce shift register
	signal WaitCount: STD_LOGIC_VECTOR(2 downto 0); -- timesetting, wait counter for auto-inc/dec
	signal EnDebounce, EnDebounceQ, KeyPressed, KeyPressedQ, KeyPulse, EnAutoIncDec, EnAutoIncDecQ, AutoIncDec: STD_LOGIC; -- timesetting, signals
	signal EnDisplayMux: STD_LOGIC_VECTOR(1 downto 0); -- signal used to mux 4-digit display
	signal TimeTick, CountUp, CountDown, Speaker, EnSpeaker: STD_LOGIC; -- diverse 
begin
	-- DEBUG ONLY
	-- 
	-- END OF DEBUG ONLY


	-- assign I/Os to signals	
	P1 <= DS_data(0);
	P2 <= DS_data(1);
	P3 <= DS_data(2);
	P4 <= DS_data(3);
	P5 <= DS_data(4);
	P6 <= DS_data(5);
	P7 <= DS_data(6);
	P8 <= DS_DP;
	P9 <= DS_enable(0);
	P10 <= DS_enable(1);
	P11 <= DS_enable(2);
	P12 <= DS_enable(3);
	P16 <= Speaker;
	SW1 <= P17; -- SW1 --> INCREMENT
	SW2 <= P18; -- SW2 --> DECREMENT
	SW3 <= P19; -- SW3 --> ENABLE TIME SETTING
	SW4 <= P20; -- SW4 --> ENABLE SPEAKER


	-- enable/disable speaker
	EnableSpeaker: process(SW4, EnSpeaker)
	begin
		if (SW4='0') then
			Speaker <= EnSpeaker;		
		else
			Speaker <= '0'; -- NPN transistor, off
		end if;
	end process;
	
	
	-- generate 1/60 Hz clock and sub-frequencies, up counter
	TimeCounter: process (CLK, COUNT)
	begin
		if CLK='1' and CLK'event then			 
			if (TimeTick='1') then -- minute elapsed      
		      COUNT <= "0000000000000000000000000000"; -- reset COUNT
			else
	         COUNT <= COUNT + 1; -- increment counter
	      end if;
	   end if;
		if COUNT="1110010011100001110000000000" then -- minute elapsed (4MHz0000)  
       	-- 4MHz0000 crystal: 60 * 4000000 counts == 1 minute, reset counter, generate 1/60 Hz signal
			TimeTick <= '1';
		else
			TimeTick <= '0';
		end if;
		-- bit position frequency table for 4MHz0000 crystal:
		-- 0	  1 2		  3	4	 5	 6	 7	 8	9 10		11	 12  13	14	 15 16 17 18 19 20 21
		-- 2MHz 1 500KHz 250 125 62 31 15 7 4 1953Hz 977 488 244 122 61 31 15 8  4  2  1	
		EnSpeaker <= COUNT(11); -- generate 977Hz signal for speaker -> jitter once every TimeTick
		EnAutoIncDec <= COUNT(18); -- generate 8Hz signal for AutoIncDec wait counter -> jitter once every TimeTick
		EnDebounce <= COUNT(13); -- generate 244Hz signal (-> max. 4ms debounce time) for debouncing input buttons -> jitter once every TimeTick
		EnDisplayMux(1 downto 0) <= COUNT(12 downto 11); -- generate 488Hz signal for display mux -> jitter once every TimeTick
	end process;
 					

	-- buttons SW1 and SW2, debounce
	buttons_debounce: process(CLK,DebounceSR(0),DebounceSR(1))
	begin
		if (clk='1') and (clk'event) then
			EnDebounceQ <= EnDebounce;								
			if (EnDebounce = '1') and (EnDebounceQ = '0') then -- at rising edge
				DebounceSR(0) <= (not SW1) or (not SW2);
				DebounceSR(1) <= DebounceSR(0);
			else
				DebounceSR(0) <= DebounceSR(0);
				DebounceSR(1) <= DebounceSR(1);
			end if;
		end if;
		KeyPressed <= DebounceSR(0) and DebounceSR(1);
	end process;


	-- buttons SW1 and SW2, generate pulse
	buttons_pulse: process(CLK,KeyPressed,KeyPressedQ)
	begin
		if (clk='1') and (clk'event) then
			KeyPressedQ <= KeyPressed;		
		end if;
		KeyPulse <= KeyPressed and (not KeyPressedQ);
	end process;


	-- generate signal AutoInDec for fast clock setting
	Clock_AutoIncDec: process(CLK)
	begin	
		if (clk='1') and (clk'event) then
			EnAutoIncDecQ <= EnAutoIncDec;		
			if KeyPressed = '0' then
				WaitCount <= "000";
			elsif EnAutoIncDec='1' and EnAutoIncDecQ='0' and WaitCount /= "111" then
				WaitCount <= WaitCount + 1;
			else
				WaitCount <= WaitCount;
			end if;
			if WaitCount = "111" then
				if (EnDebounce = '1') and (EnDebounceQ = '0') then -- at rising edge
					AutoIncDec <= '1';
				else
					AutoIncDec <= '0';
				end if;
			else
				AutoIncDec <= '0';
			end if;	
		 end if;		
	end process;


	-- generate signal CountUp
	proc_countup: process(clk) -- KeyPulse, AutoIncDec, SW1, SW3, TimeTick)
	begin
		if (clk='1') and (clk'event) then -- if line is deleted, design does not fit anymore !?
			CountUp <= ((KeyPulse or AutoIncDec) and (not SW1) and (not SW3)) or TimeTick;	
		end if; -- if line is deleted, design does not fit anymore !?
	end process;

	
	-- generate signal CountDown
	proc_countdown: process(KeyPulse, AutoIncDec, SW2, SW3)
	begin
		CountDown <= ((KeyPulse or AutoIncDec) and (not SW2) and (not SW3));			
	end process;


	-- count lo digit of minutes, 0 to 9, 4-bit up/down counter
	CountMinutes: process (CLK, nReset) 
	begin
		if nReset='0' then
				MinL <= x"0";	
				MinHinc <= '0';
				MinHdec <= '0';
	   elsif CLK='1' and CLK'event then
         if CountUp='1' and CountDown='0' then
				MinHdec <= '0';
				if MinL = x"9" then
					MinL <= x"0";
					MinHinc <= '1';
				else	  
            	MinL <= MinL + 1;
					MinHinc <= '0';
				end if;
         elsif CountDown='1' and CountUp='0' then
				MinHinc <= '0';
				if MinL = x"0" then
					MinL <= x"9";
					MinHdec <= '1';
				else
            	MinL <= MinL - 1;
					MinHdec <= '0';
				end if;
			else
				MinL <= MinL;	
				MinHinc <= '0';
				MinHdec <= '0';
         end if; 	      
	   end if;
	end process;


	-- count high digit of minutes, 0 to 5, 3-bit up/down counter
	Count10Minutes: process (CLK, nReset) 
	begin
		if nReset='0' then
				MinH <= "000";	
				HourLinc <= '0';
				HourLdec <= '0';	
	   elsif CLK='1' and CLK'event then
         if MinHinc='1' then
				HourLdec <= '0';
				if MinH = "101" then
					MinH <= "000";
					HourLinc <= '1';
				else	  
            	MinH <= MinH + 1;
					HourLinc <= '0';
				end if;
         elsif MinHdec='1' then
				HourLinc <= '0';
				if MinH = "000" then
					MinH <= "101";
					HourLdec <= '1';
				else
            	MinH <= MinH - 1;
					HourLdec <= '0';
				end if;
			else
				MinH <= MinH;
				HourLinc <= '0';
				HourLdec <= '0';	
         end if; 	      
	   end if;
	end process;


	-- count lo digit of hours, 0 to 9, 4-bit up/down counter
	CountHours: process (CLK, nReset) 
	begin
		if nReset='0' then
				HourL <= x"0";	
				HourHinc <= '0';
				HourHdec <= '0';	
	   elsif CLK='1' and CLK'event then
         if HourLinc='1' then
				HourHdec <= '0';
				if (HourL = x"9") or ((HourL = x"3") and (HourH = "10")) then
					HourL <= x"0";
					HourHinc <= '1';
				else	  
            	HourL <= HourL + 1;
					HourHinc <= '0';
				end if;
         elsif HourLdec='1' then
				HourHinc <= '0';
				if HourL = x"0" and (HourH /= "00") then
					HourL <= x"9";
					HourHdec <= '1';
				elsif HourL = x"0" and (HourH = "00") then
					HourL <= x"3";
					HourHdec <= '1';
				else
            	HourL <= HourL - 1;
					HourHdec <= '0';
				end if;
			else
				HourL <= HourL;	
				HourHinc <= '0';
				HourHdec <= '0';
         end if; 	      
	   end if;
	end process;


	-- count high digit of hours, 0 to 2, 2-bit up/down counter
	Count10Hours: process (CLK, nReset) 
	begin
		if nReset='0' then
				HourH <= "00";	
	   elsif CLK='1' and CLK'event then
         if HourHinc='1' then
				if HourH = "10" then
					HourH <= "00";
				else	  
            	HourH <= HourH + 1;
				end if;
         elsif HourHdec='1' then
				if HourH = "00" then
					HourH <= "10";
				else
            	HourH <= HourH - 1;
				end if;
			else
				HourH <= HourH;
         end if; 	      
	   end if;
	end process;
	

 	-- 4-digit 7-segment common cathode LED display, data mux, enable digit, enable DP
	display_mux: process (EnDisplayMux, MinL, MinH, HourL, HourH)
	begin
	   case EnDisplayMux is
	      when "00" =>
				DS_dataBin <= MinL; -- digit DS1, LSB
				DS_DP <= '0'; -- DP not enabled
				DS_enable <= "0001"; -- digit DS1, LSB	
	      when "01" =>
				DS_dataBin(2 downto 0) <= MinH; -- digit DS2
				DS_dataBin(3) <= '0'; 
				DS_DP <= '0'; -- DP not enabled
				DS_enable <= "0010"; -- digit DS2				
	      when "10" =>
				DS_dataBin <= HourL; -- digit DS3
				DS_DP <= '1'; -- DP enabled
				DS_enable <= "0100"; -- digit DS3				
	      when "11" =>
				DS_dataBin(1 downto 0) <= HourH; -- digit DS4, MSB
				DS_dataBin(3 downto 2) <= "00"; 
				DS_DP <= '0'; -- DP not enabled
				if HourH = "00" then
					DS_enable <= "0000"; -- digit DS4, MSB, disabled, supress leading zero
				else
					DS_enable <= "1000"; -- digit DS4, MSB, enabled	
				end if;
	      when others => NULL;
	   end case;	
	end process;
 

	--HEX-to-seven-segment decoder for 7-segment common cathode LED display
	display_decode: process(DS_dataBin)
	begin
		case DS_dataBin is
			-- segment encoding, bit order:
			--      0
			--     ---  
			--  5 |   | 1
			--     ---   <- 6
			--  4 |   | 2
			--     ---
			--      3
			when "0000" =>
				DS_data <= "0111111"; -- 0
	      when "0001" =>
				DS_data <= "0000110"; -- 1
	      when "0010" =>
				DS_data <= "1011011"; -- 2
	      when "0011" =>
				DS_data <= "1001111"; -- 3
	      when "0100" =>
				DS_data <= "1100110"; -- 4
	      when "0101" =>
				DS_data <= "1101101"; -- 5
	      when "0110" =>
				DS_data <= "1111101"; -- 6
	      when "0111" =>
				DS_data <= "0000111"; -- 7
	      when "1000" =>
				DS_data <= "1111111"; -- 8
			when "1001" =>
				DS_data <= "1101111"; -- 9
      	when others =>
				DS_data <= "0000000"; -- 0
   	end case;
	end process;

 
end Behavioral;
